//_______________________
// Object EchoScrollEventWindow

/**
 * Static object/namespace for ScrollEventWindow support.
 * This object/namespace should not be used externally.
 */
EchoScrollEventWindow = {
    
    iTimeoutId:null,
    
    /**
     * Static object/namespace for ContentPane MessageProcessor 
     * implementation.
     */
    MessageProcessor: {
    
        /**
         * MessageProcessor process() implementation 
         * (invoked by ServerMessage processor).
         *
         * @param messagePartElement the <code>message-part</code> element to process.
         */
        process: function(messagePartElement) {
            for (var i = 0; i < messagePartElement.childNodes.length; ++i) {
                if (messagePartElement.childNodes[i].nodeType == 1) {
                    switch (messagePartElement.childNodes[i].tagName) {
                    case "init":
                        EchoScrollEventWindow.MessageProcessor.processInit(messagePartElement.childNodes[i]);
                        break;
                    case "dispose":
                        EchoScrollEventWindow.MessageProcessor.processDispose(messagePartElement.childNodes[i]);
                        break;
                    case "scroll-horizontal":
                        EchoScrollEventWindow.MessageProcessor.processScroll(messagePartElement.childNodes[i]);
                        break;
                    case "scroll-vertical":
                        EchoScrollEventWindow.MessageProcessor.processScroll(messagePartElement.childNodes[i]);
                        break;
                    }
                }
            }
        },
        
        /**
         * Processes a <code>dispose</code> message to finalize the state of a
         * selection component that is being removed.
         *
         * @param disposeMessageElement the <code>dispose</code> element to process
         */
        processDispose: function(disposeMessageElement) {
            for (var item = disposeMessageElement.firstChild; item; item = item.nextSibling) {
                var elementId = item.getAttribute("eid");
                var divElement = document.getElementById(elementId);
                if (!divElement) {
                    continue;
                }
                var contentElement = EchoScrollEventWindow.findContentElement(divElement);
                if (!contentElement) {
                    continue;
                }
                EchoEventProcessor.removeHandler(contentElement, "scroll");
            }
        },
        
        /**
         * Processes an <code>init</code> message to initialize the state of a 
         * selection component that is being added.
         *
         * @param initMessageElement the <code>init</code> element to process
         */
        processInit: function(initMessageElement) {
            for (var item = initMessageElement.firstChild; item; item = item.nextSibling) {
                var elementId = item.getAttribute("eid");
                var divElement = document.getElementById(elementId);
                var contentElement = EchoScrollEventWindow.findContentElement(divElement);
        
                var horizontalScroll = item.getAttribute("horizontal-scroll");
                var verticalScroll = item.getAttribute("vertical-scroll");
                if (contentElement) {
                    EchoEventProcessor.addHandler(contentElement, "scroll", "EchoScrollEventWindow.processScroll");
                    
                    if (horizontalScroll) {
                        contentElement.scrollLeft = parseInt(horizontalScroll, 10);
                    }
                    if (verticalScroll) {
                        contentElement.scrollTop = parseInt(verticalScroll, 10);
                    }
                }
                EchoDomUtil.addEventListener(divElement, "mouseup", EchoScrollEventWindow.processMouseUp, false);
                EchoDomUtil.addEventListener(divElement, "mousemove", EchoScrollEventWindow.processMouseUp, false);
            }
        },
        
        /**
         * Processes a <code>scroll</code> directive to update the scrollbar positions
         * of the component.
         *
         * @param scrollMessageElement the <code>scroll</code> element to process
         */
        processScroll: function(scrollMessageElement) {
            var elementId = scrollMessageElement.getAttribute("eid");
            var position = parseInt(scrollMessageElement.getAttribute("position"), 10);
        
            var divElement = document.getElementById(elementId);
            var contentElement = EchoScrollEventWindow.findContentElement(divElement);
            if (!contentElement) {
                return;
            }
            
            if (position < 0) {
                position = 1000000;
            }
            
            if (scrollMessageElement.nodeName == "scroll-horizontal") {
                contentElement.scrollLeft = position;
            } else if (scrollMessageElement.nodeName == "scroll-vertical") {
                EchoDebugManager.consoleWrite("scrolling to: " + position);
                contentElement.scrollTop = position;
            }
        }
    },
        
    /**
     * Finds the content element of a ContentPane.
     * The content element is the element containing a child
     * that is NOT a FloatingPane.  (ContentPanes are limited to one
     * non-FloatingPane child).
     * 
     * @param element the root ContentPane DIV element to search.
     * @return the content DIV element, if found
     */
    findContentElement: function(element) {
        for (var child = element.firstChild; child; child = child.nextSibling) {
            if (child.id && child.id.indexOf("_content") != -1) {
                return child;
            }
        }
        return null;
    },
    
    /**
     * Processes a scrollbar adjustment event.
     *
     * @param echoEvent the event, preprocessed by the 
     *        <code>EchoEventProcessor</code>
     */
    processMouseUp: function(echoEvent) {
        if (!EchoClientEngine.verifyInput(echoEvent.registeredTarget)) {
            return;
        }
        if (EchoScrollEventWindow.iTimeoutId) {
        	window.clearTimeout(EchoScrollEventWindow.iTimeoutId);
        }
        EchoScrollEventWindow.iTimeoutId = setTimeout("EchoServerTransaction.connect()", 750);
    },
    
    /**
     * Processes a scrollbar adjustment event.
     *
     * @param echoEvent the event, preprocessed by the 
     *        <code>EchoEventProcessor</code>
     */
    processScroll: function(echoEvent) {
        if (!EchoClientEngine.verifyInput(echoEvent.registeredTarget)) {
            return;
        }
        
        var componentId = EchoDomUtil.getComponentId(echoEvent.registeredTarget.id);
        
        EchoClientMessage.setPropertyValue(componentId, "horizontalScroll", echoEvent.registeredTarget.scrollLeft + "px");
        EchoClientMessage.setPropertyValue(componentId, "verticalScroll", echoEvent.registeredTarget.scrollTop + "px");
        
        if (EchoScrollEventWindow.iTimeoutId) {
        	window.clearTimeout(EchoScrollEventWindow.iTimeoutId);
        }
        EchoScrollEventWindow.iTimeoutId = setTimeout("EchoServerTransaction.connect()", 750);
    }
};
